// src/ai/flows/dynamic-faq-generation.ts
'use server';

/**
 * @fileOverview Generates dynamic FAQs for a given course curriculum.
 *
 * - generateFaq - A function that generates FAQs based on the provided curriculum.
 * - GenerateFaqInput - The input type for the generateFaq function.
 * - GenerateFaqOutput - The return type for the generateFaq function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const GenerateFaqInputSchema = z.object({
  curriculum: z
    .string()
    .describe('The week-by-week curriculum of the course.'),
});
export type GenerateFaqInput = z.infer<typeof GenerateFaqInputSchema>;

const GenerateFaqOutputSchema = z.object({
  faq: z.array(z.object({
    question: z.string().describe('The question about the course.'),
    answer: z.string().describe('The answer to the question.'),
  })).describe('An array of questions and answers about the course.'),
});
export type GenerateFaqOutput = z.infer<typeof GenerateFaqOutputSchema>;

export async function generateFaq(input: GenerateFaqInput): Promise<GenerateFaqOutput> {
  return generateFaqFlow(input);
}

const generateFaqPrompt = ai.definePrompt({
  name: 'generateFaqPrompt',
  input: {schema: GenerateFaqInputSchema},
  output: {schema: GenerateFaqOutputSchema},
  prompt: `You are an expert in generating FAQs for online courses.

  Based on the curriculum provided, create a list of frequently asked questions and their answers.
  The questions should cover topics such as course duration, format, prerequisites, and what students will learn.

  Curriculum: {{{curriculum}}}
  `,
});

const generateFaqFlow = ai.defineFlow(
  {
    name: 'generateFaqFlow',
    inputSchema: GenerateFaqInputSchema,
    outputSchema: GenerateFaqOutputSchema,
  },
  async input => {
    const {output} = await generateFaqPrompt(input);
    return output!;
  }
);
