
"use client";

import { courses } from '@/lib/courses';
import Link from 'next/link';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle, CardDescription, CardFooter } from '@/components/ui/card';
import { ArrowRight, BookOpen, BrainCircuit, Cloud, ShieldCheck, Briefcase, Bot, ThumbsUp, Users, Zap, Search } from 'lucide-react';
import { Badge } from '@/components/ui/badge';
import Image from 'next/image';
import React from 'react';
import { Carousel, CarouselContent, CarouselItem, CarouselNext, CarouselPrevious } from "@/components/ui/carousel";
import Autoplay from "embla-carousel-autoplay";

const iconMap: { [key: string]: React.ElementType } = {
  'cloud-computing': Cloud,
  'data-science-ai': BrainCircuit,
  'data-analysis': BookOpen,
  'cyber-security': ShieldCheck,
};

const sliderImages = [
    "/slider1.png",
    "/slider2.png",
    "/slider3.png"
];

const valueProps = [
    {
        icon: Zap,
        title: "Innovation-Driven",
        description: "We are a forward-thinking startup constantly exploring new technologies to deliver cutting-edge solutions."
    },
    {
        icon: BrainCircuit,
        title: "Unique Blend of Expertise",
        description: "We combine product development, expert consultancy, and skill empowerment to provide holistic value."
    },
    {
        icon: Cloud,
        title: "Remote-First, Global Reach",
        description: "Our remote-first model allows us to work with talent and clients from anywhere in the world."
    },
    {
        icon: Search,
        title: "Proven Mentorship",
        description: "We offer career support and guidance that has been trusted by professionals and businesses alike."
    }
]

export default function Home() {
  const plugin = React.useRef(
    Autoplay({ delay: 5000, stopOnInteraction: true })
  );

  return (
    <div className="flex flex-col min-h-[100dvh]">
      <section className="relative w-full h-[70vh] md:h-[90vh]">
        <Carousel
          plugins={[plugin.current]}
          className="w-full h-full"
          onMouseEnter={plugin.current.stop}
          onMouseLeave={plugin.current.reset}
          opts={{
              loop: true,
          }}
        >
          <CarouselContent>
            {sliderImages.map((src, index) => (
              <CarouselItem key={index}>
                <div className="relative w-full h-[70vh] md:h-[90vh]">
                  <Image
                    src={src}
                    alt={`Slider Image ${index + 1}`}
                    fill
                    priority={index === 0}
                    className="object-cover"
                  />
                  <div className="absolute inset-0 bg-black/40" />
                </div>
              </CarouselItem>
            ))}
          </CarouselContent>
          <div className="absolute inset-x-0 top-1/2 -translate-y-1/2 z-20 flex items-center justify-between px-4">
            <CarouselPrevious className="static" />
            <CarouselNext className="static" />
          </div>
        </Carousel>

        <div className="absolute inset-0 z-10 flex items-center justify-center text-center text-white pointer-events-none">
            <div className="container px-4 md:px-6">
                <div className="flex flex-col items-center space-y-6 animate-hero-text-up">
                    <h1 className="text-4xl font-bold tracking-tighter sm:text-5xl xl:text-6xl/none font-headline">
                    Empowering Businesses & Individuals with Innovative Tech Solutions and Skills.
                    </h1>
                    <p className="max-w-[800px] text-lg md:text-xl">
                    From building products to delivering training and consultancy in AI, Cloud, Data, and Digital Transformation — we help you grow.
                    </p>
                    <div className="flex flex-col gap-4 min-[400px]:flex-row pointer-events-auto">
                    <Button asChild size="lg">
                        <Link href="/services/software-development">
                        View Services
                        </Link>
                    </Button>
                    <Button asChild size="lg" variant="secondary">
                        <Link href="#courses">
                        Explore Courses
                        </Link>
                    </Button>
                    </div>
                </div>
            </div>
        </div>
      </section>

      <section id="about" className="w-full py-12 md:py-24 lg:py-32 bg-background">
        <div className="container px-4 md:px-6">
            <div className="grid items-center gap-10 lg:grid-cols-2 lg:gap-16">
                <div className="space-y-4">
                    <h2 className="text-3xl font-bold tracking-tighter sm:text-4xl font-headline">About RiteClick Technology</h2>
                    <p className="text-muted-foreground md:text-xl/relaxed">
                    RiteClick Technology is a forward-thinking tech startup providing consultancy, digital solutions, and career-focused training. We combine innovation, education, and execution to deliver value for businesses and individuals worldwide.
                    </p>
                </div>
                <div className="relative h-64 lg:h-96 rounded-xl overflow-hidden">
                   <Image 
                    src="/about.png"
                    alt="About RiteClick Technology"
                    fill
                    className="object-cover"
                    data-ai-hint="startup team"
                   />
                </div>
            </div>
        </div>
      </section>
      
      <section id="offerings" className="w-full py-12 md:py-24 lg:py-32 bg-secondary">
        <div className="container px-4 md:px-6">
            <div className="text-center mb-12">
                <h2 className="text-3xl font-bold tracking-tighter sm:text-5xl font-headline">Our Core Offerings</h2>
                <p className="max-w-[900px] mx-auto text-muted-foreground md:text-xl/relaxed mt-4">
                    We provide a unique blend of services, products, and training to drive growth and innovation.
                </p>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <Card className="flex flex-col text-center items-center">
                    <CardHeader>
                        <div className="mx-auto bg-primary/10 rounded-full p-4 w-fit mb-4">
                            <Briefcase className="w-10 h-10 text-primary"/>
                        </div>
                        <CardTitle className="font-headline text-2xl">Consultancy & Services</CardTitle>
                    </CardHeader>
                    <CardContent className="flex-grow">
                        <p className="text-muted-foreground">Expert guidance in AI, cloud, data analytics, and software development to solve your most complex business challenges.</p>
                    </CardContent>
                    <CardFooter>
                        <Button asChild>
                            <Link href="/services/software-development">Explore Services</Link>
                        </Button>
                    </CardFooter>
                </Card>
                <Card className="flex flex-col text-center items-center">
                    <CardHeader>
                        <div className="mx-auto bg-primary/10 rounded-full p-4 w-fit mb-4">
                            <Bot className="w-10 h-10 text-primary"/>
                        </div>
                        <CardTitle className="font-headline text-2xl">Products</CardTitle>
                    </CardHeader>
                    <CardContent className="flex-grow">
                        <p className="text-muted-foreground">AI-powered tools, business applications, and digital products currently under development to automate and enhance your workflow.</p>
                    </CardContent>
                    <CardFooter>
                        <Button asChild>
                            <Link href="/products/boostly">View Products</Link>
                        </Button>
                    </CardFooter>
                </Card>
            </div>
        </div>
      </section>
      
       <section id="why-us" className="w-full py-12 md:py-24 lg:py-32 bg-background">
        <div className="container px-4 md:px-6">
            <div className="text-center mb-12">
                <h2 className="text-3xl font-bold tracking-tighter sm:text-5xl font-headline">Why Work With RiteClick?</h2>
                <p className="max-w-[900px] mx-auto text-muted-foreground md:text-xl/relaxed mt-4">
                    We're more than a tech company. We're a partner in growth, innovation, and empowerment.
                </p>
            </div>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8">
               {valueProps.map((prop, index) => {
                   const Icon = prop.icon;
                   return (
                     <Card key={index} className="text-center transition-transform transform hover:-translate-y-2">
                        <CardHeader>
                            <div className="mx-auto bg-primary/10 rounded-full p-3 w-fit">
                                <Icon className="h-8 w-8 text-primary" />
                            </div>
                            <CardTitle className="mt-4 font-headline text-xl">{prop.title}</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <p className="text-muted-foreground">{prop.description}</p>
                        </CardContent>
                     </Card>
                   )
               })}
            </div>
        </div>
      </section>

      <section id="courses" className="w-full py-12 md:py-24 lg:py-32 bg-secondary">
        <div className="container px-4 md:px-6">
          <div className="flex flex-col items-center justify-center space-y-4 text-center">
            <div className="space-y-2">
              <h2 className="text-3xl font-bold tracking-tighter sm:text-5xl font-headline">Our Courses</h2>
              <p className="max-w-[900px] text-muted-foreground md:text-xl/relaxed lg:text-base/relaxed xl:text-xl/relaxed">
                Choose from a range of high-demand tech fields and start your journey with our expert-led virtual training.
              </p>
            </div>
          </div>
          <div className="mx-auto grid grid-cols-1 gap-6 py-12 sm:grid-cols-2 lg:grid-cols-4">
            {courses.map((course) => {
              const Icon = iconMap[course.slug] || BookOpen;
              return (
                <Card key={course.slug} className="flex flex-col overflow-hidden transition-transform transform hover:-translate-y-2">
                  <CardHeader>
                    <div className="flex items-center gap-4">
                      <Icon className="w-10 h-10 text-primary" />
                      <CardTitle className="font-headline">{course.title}</CardTitle>
                    </div>
                  </CardHeader>
                  <CardContent className="flex-grow">
                    <CardDescription>{course.tagline}</CardDescription>
                    <Badge variant="outline" className="mt-4">Beginner Friendly</Badge>
                  </CardContent>
                  <CardFooter>
                    <Button asChild className="w-full">
                      <Link href={`/courses/${course.slug}`}>
                        Learn More <ArrowRight className="ml-2 h-4 w-4" />
                      </Link>
                    </Button>
                  </CardFooter>
                </Card>
              );
            })}
          </div>
        </div>
      </section>

       <section id="cta" className="w-full py-12 md:py-24 lg:py-32 bg-background">
        <div className="container px-4 md:px-6 text-center">
            <h2 className="text-3xl font-bold tracking-tighter sm:text-4xl font-headline">Ready to build the future together?</h2>
            <p className="max-w-[600px] mx-auto text-muted-foreground md:text-xl mt-4">
                Whether you're scaling your business or starting your tech career, we're here to help you succeed.
            </p>
            <div className="flex flex-col gap-4 min-[400px]:flex-row justify-center mt-8">
                <Button asChild size="lg" variant="secondary">
                    <Link href="#courses">
                    Enroll Now (For Learners)
                    </Link>
                </Button>
            </div>
        </div>
      </section>
    </div>
  );
}
